/*
 * OODrawUtils.java
 *
 * Created on February 19, 2003, 2:37 PM
 *
 * Copyright 2003 Danny Brewer
 * Anyone may run this code.
 * If you wish to modify or distribute this code, then
 *  you are granted a license to do so only under the terms
 *  of the Gnu Lesser General Public License.
 * See:  http://www.gnu.org/licenses/lgpl.html
 */

package nom.DannyBrewer.OOoUno;

import nom.DannyBrewer.utils.ExceptionHandler;


//----------------------------------------------------------------------
//  UNO imports
//----------------------------------------------------------------------

// base of all interfaces
import com.sun.star.uno.XInterface;

// access the implementations via names
import com.sun.star.comp.servicemanager.ServiceManager;


import com.sun.star.lang.XMultiServiceFactory;
import com.sun.star.lang.XMultiComponentFactory;
import com.sun.star.connection.XConnector;
import com.sun.star.connection.XConnection;

import com.sun.star.bridge.XUnoUrlResolver;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XInterface;
import com.sun.star.uno.XNamingService;
import com.sun.star.uno.XComponentContext;


//----------------------------------------------------------------------
//  OpenOffice.org imports
//----------------------------------------------------------------------

// staroffice interfaces to provide desktop and componentloader
// and components i.e. spreadsheets, writerdocs etc.
import com.sun.star.frame.XDesktop;
import com.sun.star.frame.XComponentLoader;

// additional classes required
import com.sun.star.sheet.*;
import com.sun.star.container.*;
import com.sun.star.table.*;
import com.sun.star.beans.*;
import com.sun.star.style.*;
import com.sun.star.lang.*;
import com.sun.star.drawing.*;
import com.sun.star.text.XText;
import com.sun.star.awt.*;



/**
 *
 * @author  danny brewer
 */
public class OODrawUtils {
    private OODrawUtils() { }
    
    
    
    //----------------------------------------------------------------------
    //  Sugar coated access to pages on a drawing document.
    //	The first page of a drawing is page zero.
    //----------------------------------------------------------------------
    
    // How many pages are on a drawing?
    
    public static int getNumDrawPages( Object drawDoc ) {
        XDrawPages drawDoc_XDrawPages = drawDoc_getXDrawPages( drawDoc );
        return drawDoc_XDrawPages.getCount();
    }
    public static int getNumDrawPages( XDrawPages drawDoc_XDrawPages ) {
        return drawDoc_XDrawPages.getCount();
    }
    
    
    // Obtain a page from a drawing.
    // Return null if not successful.
    
    public static XDrawPage getDrawPageByIndex( Object drawDoc, int pageIndex ) {
        XDrawPages drawDoc_XDrawPages = drawDoc_getXDrawPages( drawDoc );
        return getDrawPageByIndex( drawDoc_XDrawPages, pageIndex );
    }
    public static XDrawPage getDrawPageByIndex( XDrawPages drawDoc_XDrawPages, int pageIndex ) {
        XDrawPage drawPage_XDrawPage;
        try {
            // Now ask the XIndexAccess interface to the drawPages object to get a certian page.
            Object drawPage = drawDoc_XDrawPages.getByIndex( pageIndex );
            
            // Get the right interface to the page.
            drawPage_XDrawPage = QueryInterface.XDrawPage( drawPage );
        }
        catch( Exception e ) {
            ExceptionHandler.unexpectedException( e );
            return null;
        }
        
        return drawPage_XDrawPage;
    }
    
    
    // Create a new page on a drawing.
    // Return null if not successful.
    
    public static XDrawPage insertNewPageByIndex( Object drawDoc, int pageIndex ) {
        XDrawPages drawDoc_XDrawPages = drawDoc_getXDrawPages( drawDoc );
        XDrawPage xDrawPage = drawDoc_XDrawPages.insertNewByIndex( pageIndex );
        return xDrawPage;
    }
    public static XDrawPage insertNewPageByIndex( XDrawPages drawDoc_XDrawPages, int pageIndex ) {
        XDrawPage xDrawPage = drawDoc_XDrawPages.insertNewByIndex( pageIndex );
        return xDrawPage;
    }
    
    
    // Remove a page from a drawing
    
    public static void removePageFromDrawing( Object drawDoc, int pageIndex ) {
        XDrawPages drawDoc_XDrawPages = drawDoc_getXDrawPages( drawDoc );
        XDrawPage xDrawPage = getDrawPageByIndex( drawDoc_XDrawPages, pageIndex );
        drawDoc_XDrawPages.remove( xDrawPage );
    }
    public static void removePageFromDrawing( XDrawPages drawDoc_XDrawPages, int pageIndex ) {
        XDrawPage xDrawPage = getDrawPageByIndex( drawDoc_XDrawPages, pageIndex );
        drawDoc_XDrawPages.remove( xDrawPage );
    }
    public static void removePageFromDrawing( Object drawDoc, XDrawPage xDrawPage ) {
        XDrawPages xDrawPages = drawDoc_getXDrawPages( drawDoc );
        xDrawPages.remove( xDrawPage );
    }
    public static void removePageFromDrawing( XDrawPages drawDoc_XDrawPages, XDrawPage xDrawPage ) {
        drawDoc_XDrawPages.remove( xDrawPage );
    }
    
    
    
    
    //----------------------------------------------------------------------
    //  Sugar coated access to layers of a drawing document.
    //	The first layer of a drawing is page zero.
    //----------------------------------------------------------------------
    
    public static int getNumDrawLayers( Object drawDoc ) {
        XLayerManager xLayerManager = drawDoc_getXLayerManager( drawDoc );
        return xLayerManager.getCount();
    }
    
    
    
    
    //----------------------------------------------------------------------
    //  Get useful interfaces to a drawing document.
    //----------------------------------------------------------------------
    
    // Get one of the useful interfaces from a drawing document.
    // XDrawPages gives you...
    //		XDrawPage insertNewByIndex( int pageIndex )
    //		void remove( XDrawPage drawPage )
    // Since XDrawPages includes XIndexAccess, you also get...
    //		int getCount()
    //		Object getByIndex( long index )
    // Since XIndexAccess includes XElementAccess, you also get...
    //		type getElementType()
    //		boolean hasElements()
    public static XDrawPages drawDoc_getXDrawPages( Object drawDoc ) {
        // Get a different interface to the drawDoc.
        // The parameter passed in to us is the wrong interface to the drawDoc.
        XDrawPagesSupplier drawDoc_XDrawPagesSupplier = QueryInterface.XDrawPagesSupplier( drawDoc );
        
        // Ask the drawing document to give us it's draw pages object.
        Object drawPages = drawDoc_XDrawPagesSupplier.getDrawPages();
        
        // Get the XDrawPages interface to the object.
        XDrawPages drawPages_XDrawPages = QueryInterface.XDrawPages( drawPages );
        
        return drawPages_XDrawPages;
    }
    
    // Get one of the useful interfaces from a drawing document.
    // XLayerManager gives you...
    //		XLayer insertNewByIndex( int layerIndex )
    //		void remove( XLayer layer )
    //		void attachShapeToLayer( XShape shape, XLayer layer )
    //		XLayer getLayerForShape( XShape shape )
    // Since XLayerManager includes XIndexAccess, you also get...
    //		int getCount()
    //		Object getByIndex( long index )
    // Since XIndexAccess includes XElementAccess, you also get...
    //		type getElementType()
    //		boolean hasElements()
    // QueryInterface can also be used to get an XNameAccess from this object.
    // XNameAccess gives you...
    //		Object getByName( String name )
    //		String[] getElementNames()
    //		boolean hasByName( String name )
    public static XLayerManager drawDoc_getXLayerManager( Object drawDoc ) {
        // Get a different interface to the drawDoc.
        // The parameter passed in to us is the wrong interface to the drawDoc.
        XLayerSupplier drawDoc_XLayerSupplier = QueryInterface.XLayerSupplier( drawDoc );
        
        // Ask the drawing document to give us it's layer manager object.
        Object layerManager = drawDoc_XLayerSupplier.getLayerManager();
        
        // Get the XLayerManager interface to the object.
        XLayerManager layerManager_XLayerManager = QueryInterface.XLayerManager( layerManager );
        
        return layerManager_XLayerManager;
    }
    
    
    
    //----------------------------------------------------------------------
    //  Operations on Pages
    //----------------------------------------------------------------------
    
    
    public static String getPageName( Object drawPage ) {
        // Get a different interface to the drawDoc.
        XNamed drawPage_XNamed = QueryInterface.XNamed( drawPage );
        
        return drawPage_XNamed.getName();
    }
    
    public static void setPageName( Object drawPage, String pageName ) {
        // Get a different interface to the drawDoc.
        XNamed drawPage_XNamed = QueryInterface.XNamed( drawPage );
        
        drawPage_XNamed.setName( pageName );
    }
    
    
    //  Sugar Coated property manipulation.
    
    public static void setHeight( Object obj, int height ) {
        OOUtils.setIntProperty( obj, "Height", height );
    }
    
    public static int getHeight( Object obj ) {
        return OOUtils.getIntProperty( obj, "Height" );
    }
    
    
    public static void setWidth( Object obj, int width ) {
        OOUtils.setIntProperty( obj, "Width", width );
    }
    
    public static int getWidth( Object obj ) {
        return OOUtils.getIntProperty( obj, "Width" );
    }
    
    
    
    
    //----------------------------------------------------------------------
    //  Operations on Shapes
    //----------------------------------------------------------------------
    
    public static final String SHAPE_KIND_RECTANGLE	= "com.sun.star.drawing.RectangleShape";
    public static final String SHAPE_KIND_ELLIPSE	= "com.sun.star.drawing.EllipseShape";
    public static final String SHAPE_KIND_LINE		= "com.sun.star.drawing.LineShape";
    public static final String SHAPE_KIND_TEXT		= "com.sun.star.drawing.TextShape";
    
    
    public static XShape createRectangleShape( Object drawDoc, int x, int y, int width, int height ) {
        return createShape( drawDoc, SHAPE_KIND_RECTANGLE, x, y, width, height );
    }
    public static XShape createEllipseShape( Object drawDoc, int x, int y, int width, int height ) {
        return createShape( drawDoc, SHAPE_KIND_ELLIPSE, x, y, width, height );
    }
    public static XShape createLineShape( Object drawDoc, int x, int y, int width, int height ) {
        return createShape( drawDoc, SHAPE_KIND_LINE, x, y, width, height );
    }
    public static XShape createTextShape( Object drawDoc, int x, int y, int width, int height ) {
        return createShape( drawDoc, SHAPE_KIND_TEXT, x, y, width, height );
    }
    
    
    
    // Return null if not successful.
    // Possible values for kind are:
    //		com.sun.star.drawing.RectangleShape
    //		com.sun.star.drawing.EllipseShape
    //		com.sun.star.drawing.LineShape
    //		com.sun.star.drawing.TextShape
    public static XShape createShape( Object drawDoc, String kind ) {
        // We need the XMultiServiceFactory interface.
        XMultiServiceFactory drawDoc_XMultiServiceFactory;
        if( drawDoc instanceof XMultiServiceFactory ) {
            // If the right interface was passed in, just typecaset it.
            drawDoc_XMultiServiceFactory = (XMultiServiceFactory) drawDoc;
        } else {
            // Get a different interface to the drawDoc.
            // The parameter passed in to us is the wrong interface to the drawDoc.
            drawDoc_XMultiServiceFactory = QueryInterface.XMultiServiceFactory( drawDoc );
        }
        
        Object shape_noInterface;
        try {
            // Ask MultiServiceFactory to create a shape.
            // Yuck, it gives back an Object with no specific interface.
            shape_noInterface = drawDoc_XMultiServiceFactory.createInstance( kind );
        }
        catch( Exception e ) {
            ExceptionHandler.unexpectedException( e );
            return null;
        }
        
        // Get a more useful interface to the shape object.
        XShape shape = QueryInterface.XShape( shape_noInterface );
        
        return shape;
    }
    
    
    public static XShape createShape( Object drawDoc, String kind, int x, int y, int width, int height ) {
        XShape shape = createShape( drawDoc, kind );
        setShapePositionAndSize( shape, x, y, width, height );
        return shape;
    }
    
    
    
    public static void setShapePositionAndSize( XShape shape, int x, int y, int width, int height ) {
        setShapePosition( shape, x, y );
        setShapeSize( shape, width, height );
    }
    
    public static void setShapePosition( XShape shape, int x, int y ) {
        com.sun.star.awt.Point position = new com.sun.star.awt.Point( x, y );
        shape.setPosition( position );
    }
    
    public static void setShapeSize( XShape shape, int width, int height ) {
        com.sun.star.awt.Size size = new com.sun.star.awt.Size( width, height );
        try {
            shape.setSize( size );
        }
        catch( com.sun.star.beans.PropertyVetoException e ) {
            ExceptionHandler.unexpectedException( e );
        }
    }
    
    
    //  Sugar Coated property manipulation.
    
    public static void setFillColor( Object obj, int rgbFillColor ) {
        OOUtils.setIntProperty( obj, "FillColor", rgbFillColor );
    }
    
    public static int getFillColor( Object obj ) {
        return OOUtils.getIntProperty( obj, "FillColor" );
    }
    
    
    
}

